//=====================================================================================
//
//	Author				:Cytron Technologies		
//	Project				:control pwm
//	Project description	:control PWM chip using PIC18F4520 
//
//=====================================================================================

#include<p18f4520.h>

//=====================================================================================
//	configuration bit
//=====================================================================================
#pragma config OSC = HSPLL
#pragma config FCMEN = OFF 
#pragma config IESO = OFF 
#pragma config PWRT = OFF 
#pragma config BOREN = OFF 
#pragma config WDT = OFF 
#pragma config MCLRE = ON	
#pragma config PBADEN = OFF 
#pragma config DEBUG = OFF
#pragma config XINST = OFF
#pragma config LVP = OFF 

//=====================================================================================
//	define input or ouput of board here
//	
//=====================================================================================

#define	button1		PORTBbits.RB0			//button 1 active low
#define	select1		LATBbits.LATB3			//led1 active high				
#define	select2		LATBbits.LATB2			//led2 active high				
#define	select3		LATBbits.LATB1			//led3 active high
#define iic_add		LATBbits.LATB4			//address of IIC
#define reset		LATBbits.LATB7			//reset of PWM chip, 0=reset, normally high

//=====================================================================================
//	function prototype
//=====================================================================================
void ISRHigh(void);
void ISRLow(void);
void delay(unsigned long data);
void init(void);
unsigned char read_serial(void);
void send_serial(unsigned char data);
void send_dec(unsigned long data,unsigned char num_dig);
void UART115200(void);	
void OpenIIC(void);
void WritePWM_IIC(unsigned char add, unsigned char channel, unsigned char value);
unsigned char ReadPWM_IIC(unsigned char add);
void PWMdata_IIC(void);

//=====================================================================================
//	Global Variable
//=====================================================================================

//=====================================================================================
// protocol for data send to PWM chip
//=====================================================================================
// data1 = channel of PWM
// data2 = value of PWM

//=====================================================================================
//	Main Function
//	This is the main function where program start to execute 
//=====================================================================================

void main(void)
{	
	
	init();					// call initialization function of the setting in program
	while(1)
	{	
	
		if(button1==0)					//button 1 to switch mode between play and record
		{	
			while(button1==0);			//wait button 1 to release

			reset = 0;
			delay(5000);
			reset = 1;
			UART115200();
			select1 = 1;
			select2 = 0;
			select3 = 1;
			delay(10);		//delay 29us for the PWM chip to switch to selected mode
			iic_add=1;	//slave address = 0x20
			OpenIIC();
			PWMdata_IIC();	
		}
	}
	
}

//=====================================================================================
//	functions
//  User can write all the necessary function here
//=====================================================================================
//read uart data
unsigned char read_serial(void)
{
	unsigned char temp;
	while(PIR1bits.RCIF==0);
	temp=RCREG;
	return temp;
}

//send uart data
void send_serial(unsigned char data)
{
	while (PIR1bits.TXIF == 0);
	TXREG = data;	
}

void send_dec(unsigned long data,unsigned char num_dig)
{
	
	if(num_dig>=10)
	{
		data=data%10000000000;
		send_serial(data/1000000000+0x30);
	}	
	if(num_dig>=9)
	{
		data=data%1000000000;
		send_serial(data/100000000+0x30);
	}	
	if(num_dig>=8)
	{
		data=data%100000000;
		send_serial(data/10000000+0x30);
	}	
	if(num_dig>=7)
	{
		data=data%10000000;
		send_serial(data/1000000+0x30);
	}	
	if(num_dig>=6)
	{
		data=data%1000000;
		send_serial(data/100000+0x30);
	}	
	if(num_dig>=5)
	{
		data=data%100000;
		send_serial(data/10000+0x30);
	}	
	if(num_dig>=4)
	{
		data=data%10000;
		send_serial(data/1000+0x30);
	}
	if(num_dig>=3)
	{
		data=data%1000;
		send_serial(data/100+0x30);
	}
	if(num_dig>=2)
	{
		data=data%100;
		send_serial(data/10+0x30);
	}
	if(num_dig>=1)
	{
		data=data%10;
		send_serial(data+0x30);
	}
}


void delay(unsigned long data)
{
	for( ;data>0;data-=1);
}

void UART115200(void)
{
	unsigned char temp;

	//initialize uart	
	TRISCbits.TRISC6=1;		//TX
	TRISCbits.TRISC7=1;		//RX
	SPBRG = 21;		   		// default as 9600, Fosc = 32M, [8M,51] [32M,208][10M, 64][40M,260]
	
	TXSTAbits.TX9=0;		//8/9-bit transmission, 0 = 8bit, 1=9bit
	TXSTAbits.TXEN=1;		//disable/enable transmit
	TXSTAbits.SYNC=0;		//asynchronous/synchronous mode, 0 =asyn

	TXSTAbits.BRGH=1;		//low/high speed, 1 = high speed,0, SPBRG=Fosc/(64*baud)-1, 9600=15.27, 57600=1.7

	RCSTAbits.SPEN=1;		//serial port disable/enable
	RCSTAbits.RX9=0;		//8/9-bit reception, 0 =8bit
	RCSTAbits.CREN=1;		//continuous receive disable/enable

	BAUDCON = 0b00000000;	//BAUD RATE CONTROL REGISTER
	temp=RCREG;				//clear receive buffer
	temp=RCREG;

	PIR1bits.RCIF=0;		//clear interrupt flag
	PIE1bits.RCIE=0;		//disable/enable receive interrupt
	IPR1bits.RCIP=0;		//low/high priority interrupt	
}

void init(void)
{

	// this is a initialization for set the input and output of PIC18F4520
	// tris
	TRISB=0x01;				// portB as output, RB0 as input
	TRISD=0x00;				// portD as output
	TRISC=0xff;				// 
	
	select1 = 0;
	select2 = 0;
	select3 = 0;
	
}

//iic initialization
//==========================================================================
void OpenIIC(void)
{
	//configure IIC
	TRISCbits.TRISC3=1;
	TRISCbits.TRISC4=1;
	SSPSTATbits.SMP=0;		//slew rate control (0 for high speed, 400kHz)
	SSPSTATbits.CKE=0;		//SMbus select bit
	SSPCON1bits.WCOL=0;		//write collision bit
	SSPCON1bits.SSPOV=0;	//receive overflow bit
	SSPCON1bits.SSPEN=1;	//enable SDA and SCL as serial pin
	SSPCON1bits.SSPM3=1;	
	SSPCON1bits.SSPM2=0;
	SSPCON1bits.SSPM1=0;
	SSPCON1bits.SSPM0=0;	//1000,I2C Master mode, clock = FOSC/(4 * (SSPADD + 1))
	SSPADD=0x18;			//clock (100kbps=0x18) 	
}

//iic start and send add (write mode)
//==========================================================================
void PWMWrite_IIC(unsigned char add, unsigned char channel, unsigned char value)
{
	//send IIC address
	add=add<<1;
	add=add&0b11111110;			//write mode

	PIR1bits.SSPIF=0;			//clear interrupt flag bit
	SSPCON2bits.SEN=1;			//start
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit
	SSPBUF=add;					//send add
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit

	//send PWM channel
	SSPBUF=channel;				//send data
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit	

	//send PWM value
	SSPBUF=value;				//send data
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit	

	//stop IIC
	SSPCON2bits.PEN=1;			//stop
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit	
}	


unsigned char PWMRead_IIC(unsigned char add)
{
	unsigned char pwm_value;
	
	//send IIc address
	add=add<<1;
	add=add|0b00000001;			//read mode

	PIR1bits.SSPIF=0;			//clear interrupt flag bit
	SSPCON2bits.SEN=1;			//start
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit
	
	SSPBUF=add;					//send add
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit

	//read IIC data
	SSPCON2bits.RCEN=1;			//enable receive mode
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit
	pwm_value=SSPBUF;			//read data
	
	SSPCON2bits.ACKDT=1;		//send not ack
	SSPCON2bits.ACKEN=1;		//start sending not ack		
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit
	
	//stop IIC;
	SSPCON2bits.PEN=1;			//stop
	while(PIR1bits.SSPIF==0);	//wait
	PIR1bits.SSPIF=0;			//clear interrupt flag bit	
	
	//return PWM vlaue
	return pwm_value;

}

//Example to send and read IIC data
void PWMdata_IIC(void)
{
	unsigned char data1;
	
	PWMWrite_IIC(0x20,0x00,0xf5);		
	PWMWrite_IIC(0x20,0x01,0xe5);
	PWMWrite_IIC(0x20,0x02,0xd5);
	PWMWrite_IIC(0x20,0x03,0xff);
	
	PWMWrite_IIC(0x20,'r',0x00);
	data1=PWMRead_IIC(0x20);
	send_dec(data1,3);					//send PWM value to PC serial port and display using Hyperterminal
	
	PWMWrite_IIC(0x20,'r',0x01);
	data1=PWMRead_IIC(0x20);
	send_dec(data1,3);					//send PWM value to PC serial port and display using Hyperterminal

	PWMWrite_IIC(0x20,'r',0x02);
	data1=PWMRead_IIC(0x20);
	send_dec(data1,3);					//send PWM value to PC serial port and display using Hyperterminal

	PWMWrite_IIC(0x20,'r',0x03);
	data1=PWMRead_IIC(0x20);
	send_dec(data1,3);					//send PWM value to PC serial port and display using Hyperterminal

	PWMWrite_IIC(0x20,0x05,0xd5);
	PWMWrite_IIC(0x20,0x06,0xff);
}


//=====================================================================================
//	Interrupt vector
//=====================================================================================
#pragma	code InterruptVectorHigh = 0x08
void InterruptVectorHigh(void)
{
 	_asm
		goto ISRHigh		// jump to interrupt routine
	_endasm
}
#pragma code

#pragma	code InterruptVectorLow = 0x18
void InterruptVectorLow(void)
{
 	_asm
		goto ISRLow			// jump to interrupt routine
	_endasm
}
#pragma code

//=====================================================================================
//	Interupt Service Routine
//	this a function reserved for interrupt service routine
//=====================================================================================
#pragma interrupt ISRHigh
void ISRHigh(void)
{

	
}

#pragma interrupt ISRLow
void ISRLow(void)
{
	
}
